/*
 * @(#)Relationship.java  2.0  2006-03-20
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.rer.model;

import java.beans.*;
import ch.hslu.cm.simulation.*;
import ch.randelshofer.util.*;
import java.io.IOException;
import java.util.*;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
/**
 * Relationship.
 *
 * @author Werner Randelshofer
 * @version 2.0 2006-03-20 Cleaned up.
 * <br>1.1 2006-02-21 Support for conditional cardinality added.
 * <br>1.0 2006-01-18 Created.
 */
public class Relationship extends AbstractRelationship
        implements AttributedElementListener {
    public final static int UNSPECIFIED_CARDINALITY = 0;
    public final static int MANY_CARDINALITY = -1;
    public final static int CONDITIONAL_CARDINALITY = -2;
    public final static int MANY_CONDITIONAL_CARDINALITY = -3;
    
    /*
     * 0 and negative values have special meanings.
     * 1..9999 = count
     */
    private int startCardinality = UNSPECIFIED_CARDINALITY;
    private int endCardinality = UNSPECIFIED_CARDINALITY;
    private String startLabel = RelationalERModel.labels.getString("relationshipStartLabel");
    private String endLabel = RelationalERModel.labels.getString("relationshipEndLabel");
    private boolean isEndTraversable = false;
    private boolean isStartTraversable = false;
    private RERAttribute startAttribute;
    private RERAttribute endAttribute;
    private boolean isResolved;
    
    /** Creates a new instance. */
    public Relationship() {
    }
    
    public Entity getStartEntity() {
        return (Entity) getStart();
    }
    public Entity getEndEntity() {
        return (Entity) getEnd();
    }
    
    public void setStartCardinality(int newValue) {
        int oldValue = startCardinality;
        startCardinality = newValue;
        firePropertyChange("startCardinality", oldValue, newValue);
    }
    public int getStartCardinality() {
        return startCardinality;
    }
    
    public void setEndCardinality(int newValue) {
        int oldValue = endCardinality;
        endCardinality = newValue;
        firePropertyChange("endCardinality", oldValue, newValue);
    }
    public int getEndCardinality() {
        return endCardinality;
    }
    
    public void setEndLabel(String newValue) {
        String oldValue = endLabel;
        endLabel = newValue;
        firePropertyChange("endLabel", oldValue, newValue);
    }
    public String getEndLabel() {
        return endLabel;
    }
    
    public void setStartLabel(String newValue) {
        String oldValue = startLabel;
        startLabel = newValue;
        firePropertyChange("startLabel", oldValue, newValue);
    }
    public String getStartLabel() {
        return startLabel;
    }
    
    public void setStartAttribute(RERAttribute newValue) {
        RERAttribute oldValue = startAttribute;
        startAttribute = newValue;
        firePropertyChange("startAttribute", oldValue, newValue);
    }
    public void setEndAttribute(RERAttribute newValue) {
        RERAttribute oldValue = endAttribute;
        endAttribute = newValue;
        firePropertyChange("endAttribute", oldValue, newValue);
    }
    public RERAttribute getStartAttribute() {
        return startAttribute;
    }
    public RERAttribute getEndAttribute() {
        return endAttribute;
    }
    
    public void setEndTraversable(boolean newValue) {
        boolean oldValue = isEndTraversable;
        isEndTraversable = newValue;
        firePropertyChange("endTraversable", oldValue, newValue);
    }
    public boolean isEndTraversable() {
        return isEndTraversable;
    }
    
    public void setStartTraversable(boolean newValue) {
        boolean oldValue = isStartTraversable;
        isStartTraversable = newValue;
        firePropertyChange("startTraversable", oldValue, newValue);
    }
    public boolean isStartTraversable() {
        return isStartTraversable;
    }
    
    public int getSimulatedConcept() {
        return RelationalERModel.RELATIONSHIP;
    }
    
    public boolean canConnect(SimulatedElement start, SimulatedElement end) {
        if (
                start.getSimulatedConcept() == RelationalERModel.ENTITY
                && end.getSimulatedConcept() == RelationalERModel.ENTITY
                ) {
            return true;
        }
        return false;
    }
    public boolean canConnect(SimulatedElement start) {
        return start.getSimulatedConcept() == RelationalERModel.ENTITY;
    }
    
    public Relationship clone() {
        Relationship that = (Relationship) super.clone();
        return that;
    }
    
    public void read(DOMInput in) throws IOException {
        ResourceBundleUtil labels = RelationalERModel.labels;
        isResolved = in.getAttribute("resolved", false);
        
        // Write the start of the relationship
        in.openElement("Start");
        String value = in.getAttribute("cardinality", (String) null);
        if (value == null) {
            startCardinality = UNSPECIFIED_CARDINALITY;
        } else {
            if (value.equals("c")) {
                startCardinality = CONDITIONAL_CARDINALITY;
            } else if (value.equals("m")) {
                startCardinality = MANY_CARDINALITY;
            } else if (value.equals("mc")) {
                startCardinality = MANY_CONDITIONAL_CARDINALITY;
            } else {
                startCardinality = in.getAttribute("cardinality", 1);
            }
        }
        
        startLabel = in.getAttribute("name", labels.getString("relationshipStartLabel"));
        isStartTraversable = in.getAttribute("traversable", false);
        
        in.openElement("Entity");
        setStart((Entity) in.readObject());
        in.closeElement();
        if (in.getElementCount("Attribute") == 1) {
            in.openElement("Attribute");
            setStartAttribute((RERAttribute) in.readObject());
            in.closeElement();
        }
        in.closeElement();
        
        in.openElement("End");
        value = in.getAttribute("cardinality", (String) null);
        if (value == null) {
            endCardinality = UNSPECIFIED_CARDINALITY;
        } else {
            if (value.equals("c")) {
                endCardinality = CONDITIONAL_CARDINALITY;
            } else if (value.equals("m")) {
                endCardinality = MANY_CARDINALITY;
            } else if (value.equals("mc")) {
                endCardinality = MANY_CONDITIONAL_CARDINALITY;
            } else {
                endCardinality = in.getAttribute("cardinality", 1);
            }
        }
        in.openElement("Entity");
        setEnd((Entity) in.readObject());
        in.closeElement();
        endLabel = in.getAttribute("name", labels.getString("relationshipEndLabel"));
        isEndTraversable = in.getAttribute("traversable", false);
        if (in.getElementCount("Attribute") == 1) {
            in.openElement("Attribute");
            setEndAttribute((RERAttribute) in.readObject());
            in.closeElement();
        }
        in.closeElement();
    }
    public void write(DOMOutput out) throws IOException {
        if (isResolved) {
            out.addAttribute("resolved", isResolved);
        }

        // Write start
        out.openElement("Start");
        switch (startCardinality) {
            case MANY_CARDINALITY :
                out.addAttribute("cardinality", "m");
                break;
            case CONDITIONAL_CARDINALITY :
                out.addAttribute("cardinality", "c");
                break;
            case MANY_CONDITIONAL_CARDINALITY :
                out.addAttribute("cardinality", "mc");
                break;
            case UNSPECIFIED_CARDINALITY :
                break;
            default :
                out.addAttribute("cardinality", startCardinality);
        }
        if (startLabel != null &&
                (
                isStartTraversable || ! startLabel.equals(
                RelationalERModel.labels.getString("relationshipStartLabel"))
                )
                ) {
            out.addAttribute("name", startLabel);
        }
        if (isStartTraversable) {
            out.addAttribute("traversable", isStartTraversable);
        }
        
        out.openElement("Entity");
        out.writeObject(getStart());
        out.closeElement();
        if (startAttribute != null) {
            out.openElement("Attribute");
            out.writeObject(startAttribute);
            out.closeElement();
        }
        out.closeElement();
        
        // Write end
        out.openElement("End");
        switch (endCardinality) {
            case MANY_CARDINALITY :
                out.addAttribute("cardinality", "m");
                break;
            case CONDITIONAL_CARDINALITY :
                out.addAttribute("cardinality", "c");
                break;
            case MANY_CONDITIONAL_CARDINALITY :
                out.addAttribute("cardinality", "mc");
                break;
            case UNSPECIFIED_CARDINALITY :
                break;
            default :
                out.addAttribute("cardinality", endCardinality);
        }
        if (endLabel != null &&
                (
                isEndTraversable || ! endLabel.equals(
                RelationalERModel.labels.getString("relationshipEndLabel"))
                )
                ) {
            out.addAttribute("name", endLabel);
        }
        if (isEndTraversable) {
            out.addAttribute("traversable", isEndTraversable);
        }
        out.openElement("Entity");
        out.writeObject(getEnd());
        out.closeElement();
        if (endAttribute != null) {
            out.openElement("Attribute");
            out.writeObject(endAttribute);
            out.closeElement();
        }
        out.closeElement();
    }
    
    /**
     * Handles the disconnection of a connection.
     * Override this method to handle this event.
     */
    protected void handleDisconnecting(SimulatedElement start, SimulatedElement end) {
        Entity s = (Entity) start;
        Entity e = (Entity) end;
        s.removeAttributedElementListener((AttributedElementListener) this);
        e.removeAttributedElementListener(this);
    }
    
    /**
     * Handles the connection of a connection.
     * Override this method to handle this event.
     */
    protected void handleConnecting(SimulatedElement start, SimulatedElement end) {
        Entity s = (Entity) start;
        Entity e = (Entity) end;
        s.adddAttributedElementListener(this);
        e.adddAttributedElementListener(this);
    }
    
    public void attributeAdded(AttributedElementEvent event) {
    }
    
    public void attributeChanged(AttributedElementEvent event) {
    }
    
    public void attributeRemoved(AttributedElementEvent event) {
        if (event.getAttribute() == startAttribute) {
            startAttribute = null;
            fireStateChanged();
            fireObjectRequestRemove();
            return;
        }
        if (event.getAttribute() == endAttribute) {
            endAttribute = null;
            fireStateChanged();
            fireObjectRequestRemove();
        }
        
    }
    public void nameChanged(AttributedElementEvent event) {
    }
    public void typeChanged(AttributedElementEvent event) {
    }
    
    public void setResolved(boolean newValue) {
        boolean oldValue = isResolved;
        isResolved = newValue;
        firePropertyChange("resolved", oldValue, newValue);
    }
    public boolean isResolved() {
        return isResolved;
    }
}
